<?php
/**
 * Flash Sales Prestashop module
 * 
 * @author    Prestaddons <contact@prestaddons.fr>
 * @copyright 2014 Prestaddons
 * @license
 * @link      http://www.prestaddons.fr
 */

if (!defined('_PS_VERSION_'))
	exit;

require (dirname(__FILE__).'/fsmodel.class.php');

class FlashSales extends Module
{
	/**
	 * Constructeur de la classe Flash Sales
	 */
	public function __construct()
	{
		$this->name = 'flashsales';
		$this->short_name = 'fs';
		$this->tab = 'pricing_promotion';
		$this->version = '1.9';
		$this->need_instance = 0;
		$this->ps_versions_compliancy = array('min' => '1.5', 'max' => '1.7');
		$this->controllers = array('display');
		$this->bootstrap = true;
		$this->module_key = '73f1f4bea928b0a50f7ef168bb07b4c9';

		parent::__construct();

		$this->displayName = $this->l('Flash Sales');
		$this->description = $this->l('Increase your conversion rate by creating and managing flash sales on your shop');
		$this->confirmUninstall = $this->l('Are you sure you want to uninstall the Flash Sales module?');
		$this->author = $this->l('Prestaddons');
		$this->contact = 'contact@prestaddons.fr';
	}

	/**
	 * Méthode install()
	 * 
	 * Gère l'installation du module
	 * 
	 * @return bool True si l'installation a fonctionné, false dans le cas contraire
	 */
	public function install()
	{
		$params = $this->_initFixtures();

		if (Shop::isFeatureActive())
			Shop::setContext(Shop::CONTEXT_ALL);

		if (!parent::install()
			|| !FsModel::createDB()
			|| !$this->registerHook('header')
			|| !$this->registerHook('leftColumn')
			|| !$this->registerHook('rightColumn')
			|| !$this->registerHook('extraRight')
			|| !Configuration::updateValue($this->short_name.'_font_color', '#ffffff')
			|| !Configuration::updateValue($this->short_name.'_bg_color_start', '#3A3A3A')
			|| !Configuration::updateValue($this->short_name.'_bg_color_end', '#444444')
			|| !Configuration::updateValue($this->short_name.'_product_text', rawurlencode(serialize($params['product_text'])))
			|| !Configuration::updateValue($this->short_name.'_product_font_color', '#3A3A3A')
			|| !Configuration::updateValue($this->short_name.'_disp_border', 1)
			|| !Configuration::updateValue($this->short_name.'_border_color', '#cecece')
			|| !Configuration::updateValue($this->short_name.'_meta_title', rawurlencode(serialize($params['meta_title'])))
			|| !Configuration::updateValue($this->short_name.'_meta_description', rawurlencode(serialize($params['meta_description'])))
			|| !Configuration::updateValue($this->short_name.'_banner_title', rawurlencode(serialize($params['banner_title'])))
			|| !Configuration::updateValue($this->short_name.'_banner_text', rawurlencode(serialize($params['banner_text'])))
			|| !Configuration::updateValue($this->short_name.'_banner_title_color', '#333333')
			|| !Configuration::updateValue($this->short_name.'_banner_text_color', '#333333')
			|| !Configuration::updateValue($this->short_name.'_banner_img', 'flash-sales-banner-s-1.jpg')
			|| !Configuration::updateValue($this->short_name.'_block_product_number', 2))
			return false;

		if ($this->checkPSVersion('1.6.0.0'))
		{
			if (!$this->registerHook('displayHomeTab') || !$this->registerHook('displayHomeTabContent'))
				return false;
		}

		$this->_generateCSS();

		return true;
	}

	/**
	 * Méthode _initFixtures()
	 * 
	 * Initialise tous les paramètres nécessaires à l'installation du module
	 * 
	 * @return array $params Tableau contenant les paramètres nécessaires à l'installation
	 */
	private function _initFixtures()
	{
		$id_lang_en = Language::getIdByIso('en');
		$id_lang_fr = Language::getIdByIso('fr');

		$params = array(
			'product_text' => array(
				$id_lang_en => 'End of the flash sale in:',
				$id_lang_fr => 'Fin de la vente flash dans :'
			),
			'meta_title' => array(
				$id_lang_en => 'Flash sales - '.Configuration::get('PS_SHOP_NAME'),
				$id_lang_fr => 'Ventes flash - '.Configuration::get('PS_SHOP_NAME')
			),
			'meta_description' => array(
				$id_lang_en => 'Take advantage of flash sales and get discounts on your favorite products!',
				$id_lang_fr => 'Profitez des ventes flashs et obtenez des réductions sur vos produits préférés !'
			),
			'banner_title' => array(
				$id_lang_en => 'Flash sales',
				$id_lang_fr => 'Ventes flash'
			),
			'banner_text' => array(
				$id_lang_en => '<p><strong>Take advantage of flash sales and get discounts on your favorite products!</strong></p><p>This category includes all flash sales underway.</p><p>Buy these products quickly to take advantage of great discounts!</p>',
				$id_lang_fr => '<p><strong>Profitez des ventes flashs et obtenez des r&eacute;ductions sur vos produits pr&eacute;f&eacute;r&eacute;s !</strong></p><p>Cette cat&eacute;gorie comprend toutes les ventes flashs en cours.</p><p>Achetez vite ces produits pour b&eacute;n&eacute;ficier de r&eacute;ductions exceptionnelles !</p>'
			)
		);

		return $params;
	}

	/**
	 * Méthode uninstall()
	 * 
	 * Gère la désinstallation du module
	 * 
	 * @return bool True si la désinstallation a fonctionné, false dans le cas contraire
	 */
	public function uninstall()
	{
		if (!parent::uninstall()
			|| !$this->_deleteAllSpecificPrices()
			|| !FsModel::DropDB()
			|| !Configuration::deleteByName($this->short_name.'_font_color')
			|| !Configuration::deleteByName($this->short_name.'_bg_color_start')
			|| !Configuration::deleteByName($this->short_name.'_bg_color_end')
			|| !Configuration::deleteByName($this->short_name.'_product_text')
			|| !Configuration::deleteByName($this->short_name.'_product_font_color')
			|| !Configuration::deleteByName($this->short_name.'_disp_border')
			|| !Configuration::deleteByName($this->short_name.'_border_color')
			|| !Configuration::deleteByName($this->short_name.'_meta_title')
			|| !Configuration::deleteByName($this->short_name.'_meta_description')
			|| !Configuration::deleteByName($this->short_name.'_banner_title')
			|| !Configuration::deleteByName($this->short_name.'_banner_text')
			|| !Configuration::deleteByName($this->short_name.'_banner_title_color')
			|| !Configuration::deleteByName($this->short_name.'_banner_text_color')
			|| !Configuration::deleteByName($this->short_name.'_banner_img')
			|| !Configuration::deleteByName($this->short_name.'_block_product_number'))
			return false;

		return true;
	}

	/**
	 * Méthode _postValidation()
	 * 
	 * ContrÃ´le les variables saisies dans le backoffice et définit les éventuelles erreurs à afficher
	 * 
	 * @return string HTML du résultat de la vérification (message d'erreur éventuel)
	 */
	private function _postValidation()
	{
		if (Tools::isSubmit('submit'.$this->name))
		{
			if (Tools::getValue('id_product') == '')
				$this->_postErrors[] = $this->l('You must select a product for your flash sale');
			if (!Validate::isDateFormat(Tools::getValue('date_from')))
				$this->_postErrors[] = $this->l('Invalid start date');
			if (!Validate::isDateFormat(Tools::getValue('date_to')))
				$this->_postErrors[] = $this->l('Invalid end date');
			if (!Validate::isUnsignedFloat(Tools::getValue('discount')))
				$this->_postErrors[] = $this->l('Discount must be a positive float');
			if ((Tools::getValue('unit_discount') != 'amount') && (Tools::getValue('unit_discount') != 'percentage'))
				$this->_postErrors[] = $this->l('Discount type must be a valid type');
			if (Tools::getValue('action_form') == 'add')
			{
				$id_product = Tools::getvalue('id_product');
				$id_product_attribute = Tools::getvalue('id_product_attribute');
				$specific_prices = FsModel::getByProductId((int)$id_product, (int)$id_product_attribute);
				if (count($specific_prices) > 0)
					$this->_postErrors[] = $this->l('A specific price already exists for this product / attribute. You must delete it before create a flash sale.');
			}
		}
		else if (Tools::isSubmit('submitproductpages'))
		{
			if (!Validate::isColor(Tools::getValue('font_color')))
				$this->_postErrors[] = $this->l('Font color must be a valid color');
			if (!Validate::isColor(Tools::getValue('bg_color_start')))
				$this->_postErrors[] = $this->l('First background color must be a valid color');
			if (!Validate::isColor(Tools::getValue('bg_color_end')))
				$this->_postErrors[] = $this->l('Second background color must be a valid color');
			if (!Validate::isColor(Tools::getValue('product_font_color')))
				$this->_postErrors[] = $this->l('Product font color must be a valid color');
			if (!Validate::isColor(Tools::getValue('border_color')))
				$this->_postErrors[] = $this->l('Border color must be a valid color');
		}
		else if (Tools::isSubmit('submitflashsalespage'))
		{
			if (!Validate::isColor(Tools::getValue('banner_title_color')))
				$this->_postErrors[] = $this->l('Banner title color must be a valid color');
			if (!Validate::isColor(Tools::getValue('banner_text_color')))
				$this->_postErrors[] = $this->l('Banner text color must be a valid color');
			if (isset($_FILES['banner_img']) && $_FILES['banner_img']['name'] != '')
			{
				if (!ImageManager::isCorrectImageFileExt($_FILES['banner_img']['name']))
					$this->_postErrors[] = $this->l('Banner image must be a valid image file');
			}
		}
		else if (Tools::isSubmit('submitflashsalesblock'))
		{
			if (!Validate::isUnsignedInt(Tools::getValue('block_product_number')))
				$this->_postErrors[] = $this->l('Block product number must be a valid integer');
		}
	}

	/**
	 * Méthode _postProcess()
	 * 
	 * Traitement des informations saisies dans le backoffice
	 * Traitements divers, mise à jour la base de données, définition des messages d'erreur ou de confirmation...
	 * 
	 * @return string HTML du résultat du traitement (message d'erreur ou de confirmation)
	 */
	private function _postProcess()
	{
		if (Tools::isSubmit('submit'.$this->name))
		{
			$id_product = Tools::getvalue('id_product');
			$id_product_attribute = Tools::getvalue('id_product_attribute');
			$date_from = Tools::getvalue('date_from');
			$date_to = Tools::getvalue('date_to');
			$discount = Tools::getvalue('discount');
			$unit_discount = Tools::getvalue('unit_discount');
			$status = Tools::getvalue('status');
			$id_specific_price = Tools::getvalue('id_specific_price');
			$id_currency = Tools::getvalue('id_currency');
			$id_country = Tools::getvalue('id_country');
			$id_group = Tools::getvalue('id_group');

			if (Tools::getValue('action_form') == 'add')
			{
				if ($status)
					$id_specific_price = $this->_addSpecificPrice($id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $id_currency, $id_country, $id_group);

				FsModel::insertFlashSale($id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $status, $id_specific_price, $id_currency, $id_country, $id_group);

				$this->_html .= $this->displayConfirmation($this->l('Flash sale have been added'));
			}
			else if (Tools::getValue('action_form') == 'update')
			{
				if ($status && !$id_specific_price)
					$id_specific_price = $this->_addSpecificPrice($id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $id_currency, $id_country, $id_group);
				else if ($status)
					$this->_updateSpecificPrice($id_specific_price, $id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $id_currency, $id_country, $id_group);
				else if (!$status)
					$this->_deleteSpecificPrice($id_specific_price);

				$id_flash_sale = Tools::getvalue('id_flash_sale');
				FsModel::updateFlashSale($id_flash_sale, $id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $status, $id_specific_price, $id_currency, $id_country, $id_group);

				$this->_deleteTmpCover($id_flash_sale);

				$this->_html .= $this->displayConfirmation($this->l('Flash sale have been updated'));
			}
		}
		else if (Tools::isSubmit('submitproductpages'))
		{
			$languages = Language::getLanguages(false);

			$product_text = array();
			foreach ($languages as $language)
				$product_text[$language['id_lang']] = Tools::getValue('product_text_'.$language['id_lang']);

			Configuration::updateValue($this->short_name.'_product_text', rawurlencode(serialize($product_text)));
			Configuration::updateValue($this->short_name.'_font_color', Tools::getValue('font_color'));
			Configuration::updateValue($this->short_name.'_bg_color_start', Tools::getValue('bg_color_start'));
			Configuration::updateValue($this->short_name.'_bg_color_end', Tools::getValue('bg_color_end'));
			Configuration::updateValue($this->short_name.'_product_font_color', Tools::getValue('product_font_color'));
			Configuration::updateValue($this->short_name.'_disp_border', Tools::getValue('disp_border'));
			Configuration::updateValue($this->short_name.'_border_color', Tools::getValue('border_color'));

			$this->_generateCSS();

			$this->_html .= $this->displayConfirmation($this->l('Settings have been updated'));
		}
		else if (Tools::isSubmit('submitflashsalespage'))
		{
			$languages = Language::getLanguages(false);

			if (isset($_FILES['banner_img']) && $_FILES['banner_img']['name'] != '')
			{
				// Getting the old filename stored in the database
				$old_file = Configuration::get($this->short_name.'_banner_img');

				// Getting the id_shop of the old file stored in the database
				$old_file_ext = strrchr($old_file, '.');
				$old_filename = Tools::substr($old_file, 0, Tools::strlen($old_file) - Tools::strlen($old_file_ext));
				$old_file_id_shop = Tools::substr(strrchr($old_filename, '-'), 1);

				if (file_exists(dirname(__FILE__).'/img/'.$old_file) && $old_file_id_shop == $this->context->shop->id)
					@unlink(dirname(__FILE__).'/img/'.$old_file);

				// Setting the image's name with a name contextual to the shop context
				$file_ext = strrchr($_FILES['banner_img']['name'], '.');
				$filename = Tools::substr($_FILES['banner_img']['name'], 0, Tools::strlen($_FILES['banner_img']['name']) - Tools::strlen($file_ext));
				$filename .= '-s-'.$this->context->shop->id.$file_ext;

				// Copying the image in the module directory with its new contextual name
				if (@!move_uploaded_file($_FILES['banner_img']['tmp_name'], dirname(__FILE__).'/img/'.$filename))
					$this->_html .= '<div class="alert error">'.$this->l('An error occurred while uploading the file.').'</div>';
				else
					Configuration::updateValue($this->short_name.'_banner_img', $filename);
			}

			$meta_title = array();
			$meta_description = array();
			$banner_title = array();
			$banner_text = array();

			foreach ($languages as $language)
			{
				$meta_title[$language['id_lang']] = Tools::getValue('meta_title_'.$language['id_lang']);
				$meta_description[$language['id_lang']] = Tools::getValue('meta_description_'.$language['id_lang']);
				$banner_title[$language['id_lang']] = Tools::getValue('banner_title_'.$language['id_lang']);
				$banner_text[$language['id_lang']] = Tools::getValue('banner_text_'.$language['id_lang']);
			}

			Configuration::updateValue($this->short_name.'_meta_title', rawurlencode(serialize($meta_title)));
			Configuration::updateValue($this->short_name.'_meta_description', rawurlencode(serialize($meta_description)));
			Configuration::updateValue($this->short_name.'_banner_title', rawurlencode(serialize($banner_title)));
			Configuration::updateValue($this->short_name.'_banner_text', rawurlencode(serialize($banner_text)));
			Configuration::updateValue($this->short_name.'_banner_title_color', Tools::getValue('banner_title_color'));
			Configuration::updateValue($this->short_name.'_banner_text_color', Tools::getValue('banner_text_color'));

			$this->_generateCSS();

			$this->_html .= $this->displayConfirmation($this->l('Settings have been updated'));
		}
		else if (Tools::isSubmit('submitflashsalesblock'))
		{
			Configuration::updateValue($this->short_name.'_block_product_number', Tools::getValue('block_product_number'));

			$this->_generateCSS();

			$this->_html .= $this->displayConfirmation($this->l('Settings have been updated'));
		}
	}

	/**
	 * Méthode getContent()
	 * 
	 * Gère l'administration du module dans le backoffice
	 * Dispatch vers les différentes méthodes en fonctions des cas (affichage des formulaires, des erreurs, des confirmations, ...)
	 * 
	 * @return string HTML de la partie backoffice du module
	 */
	public function getContent()
	{
		$this->_postValidation();

		if (!isset($this->_postErrors) || !count($this->_postErrors))
			$this->_postProcess();
		else
		{
			foreach ($this->_postErrors as $err)
				$this->_html .= $this->displayError($err);
		}

		if (Tools::isSubmit('addflashsales') || Tools::isSubmit('updateflashsales'))
			$this->_html .= $this->renderForm();
		else if (Tools::isSubmit('settings'))
			$this->_html .= $this->renderSettingsForm();
		else if (Tools::isSubmit('support'))
			$this->_html .= $this->renderSupportForm();
		else
		{
			if (Tools::isSubmit('deleteflashsales'))
				$this->_html .= $this->_deleteFlashSale();
			if (Tools::isSubmit('statusflashsales'))
				$this->_changeStatus();

			$this->_html .= $this->renderList();
		}

		return $this->_html;
	}

	public function renderList()
	{
		$helper = $this->initList();
		return $helper->generateList($this->getListContent((int)Configuration::get('PS_LANG_DEFAULT')), $this->fields_list);
	}

	protected function initList()
	{
		$this->fields_list = array();

		if (Shop::isFeatureActive())
			$this->fields_list['id_shop'] = array('title' => $this->l('ID Shop'), 'align' => 'center', 'type' => 'int', 'search' => false);

		$this->fields_list['id_flash_sale'] = array(
			'title' => $this->l('ID'),
			'align' => 'center',
			'class' => 'fixed-width-xs',
			'type' => 'int',
			'search' => false
		);
		$this->fields_list['image'] = array(
			'title' => $this->l('Photo'),
			'align' => 'center',
			'image' => 'p',
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['name'] = array(
			'title' => $this->l('Name'),
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['attribute_name'] = array(
			'title' => $this->l('Attribute name'),
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['currency_name'] = array(
			'title' => $this->l('Currency'),
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['country_name'] = array(
			'title' => $this->l('Country'),
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['group_name'] = array(
			'title' => $this->l('Group'),
			'orderby' => false,
			'search' => false
		);
		$this->fields_list['date_from'] = array(
			'title' => $this->l('From'),
			'type' => 'datetime',
			'align' => 'center',
			'search' => false
		);
		$this->fields_list['date_to'] = array(
			'title' => $this->l('To'),
			'type' => 'datetime',
			'align' => 'center',
			'search' => false
		);
		$this->fields_list['discount'] = array(
			'title' => $this->l('Discount'),
			'type' => 'decimal',
			'align' => 'text-right',
			'search' => false
		);
		$this->fields_list['unit_discount'] = array(
			'title' => $this->l('Unit'),
			'align' => 'text-left',
			'search' => false
		);
		$this->fields_list['status'] = array(
			'title' => $this->l('Status'),
			'active' => 'status',
			'align' => 'text-center',
			'type' => 'bool',
			'class' => 'fixed-width-sm',
			'orderby' => true,
			'search' => false
		);

		$helper = new HelperList();
		$helper->shopLinkType = '';
		$helper->simple_header = false;
		$helper->identifier = 'id_flash_sale';
		$helper->actions = array('edit', 'delete');
		$helper->show_toolbar = true;
		$helper->imageType = 'jpg';
		$helper->toolbar_btn['new'] = array(
			'href' => AdminController::$currentIndex.'&configure='.$this->name.'&add'.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
			'desc' => $this->l('Add new')
		);

		if ($this->checkPSVersion('1.6.0.0'))
		{
			$helper->toolbar_btn['cogs'] = array(
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&settings&token='.Tools::getAdminTokenLite('AdminModules'),
				'desc' => $this->l('Settings')
			);
		}
		else
		{
			$helper->toolbar_btn['new-module'] = array(
				'href' => AdminController::$currentIndex.'&configure='.$this->name.'&settings&token='.Tools::getAdminTokenLite('AdminModules'),
				'desc' => $this->l('Settings')
			);
		}

		$helper->toolbar_btn['help-new'] = array(
			'href' => AdminController::$currentIndex.'&configure='.$this->name.'&support&token='.Tools::getAdminTokenLite('AdminModules'),
			'desc' => $this->l('Support')
		);

		$helper->tpl_vars['icon'] = 'icon-flash';
		$helper->title = $this->displayName;
		$helper->table = $this->name;
		$helper->token = Tools::getAdminTokenLite('AdminModules');
		$helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
		$helper->listTotal = count($this->getListContent($this->context->cookie->id_lang));

		return $helper;
	}

	public function getListContent($id_lang)
	{
		$order_by = 'date_to';
		$order_way = 'ASC';

		if (Tools::isSubmit($this->name.'Orderby'))
		{
			$order_by = Tools::getValue($this->name.'Orderby');
			$order_way = Tools::getValue($this->name.'Orderway');
		}

		$product_list = FsModel::getFlashSales($order_by, $order_way);
		foreach ($product_list as &$product)
		{
			$product['name'] = Product::getProductName($product['id_product'], null, $id_lang);
			$cover = Image::getCover($product['id_product']);
			$product['id_image'] = $cover['id_image'];
			$attributes = Product::getAttributesParams($product['id_product'], $product['id_product_attribute']);
			$product['attribute_name'] = '';
			foreach ($attributes as $attribute)
				$product['attribute_name'] .= $attribute['name'].' - ';

			$product['attribute_name'] = Tools::substr($product['attribute_name'], 0, Tools::strlen($product['attribute_name']) - 2);
			if ($product['attribute_name'] == '')
				$product['attribute_name'] = $this->l('All');

			$product['unit_discount'] = ($product['unit_discount'] == 'amount')?$this->l('Currency Units'):$this->l('Percent');
			if (time() < strtotime($product['date_from']) || time() > strtotime($product['date_to']))
				$product['status'] = 0;

			$currency = Currency::getCurrency($product['id_currency']);
			$group = new Group($product['id_group'], $this->context->cookie->id_lang, $this->context->shop->id);

			$product['currency_name'] = ($product['id_currency'] == 0)?$this->l('All'):$currency['name'];
			$product['country_name'] = ($product['id_country'] == 0)?$this->l('All'):Country::getNameById($this->context->cookie->id_lang, $product['id_country']);
			$product['group_name'] = ($product['id_group'] == 0)?$this->l('All'):$group->name;
		}

		return $product_list;
	}

	private function _deleteFlashSale()
	{
		$id_flash_sale = Tools::getValue('id_flash_sale');
		$flash_sale = FsModel::getFlashSale($id_flash_sale);

		$this->_deleteSpecificPrice($flash_sale[0]['id_specific_price']);
		FsModel::deleteFlashSale($id_flash_sale);
		$this->_deleteTmpCover($id_flash_sale);

		return $this->displayConfirmation($this->l('Flash sale have been deleted'));
	}

	private function _changeStatus()
	{
		$id_flash_sale = Tools::getValue('id_flash_sale');
		$flash_sale = FsModel::getFlashSale($id_flash_sale);
		$status = ($flash_sale[0]['status'] == 0)?1:0;

		if ($status == 0)
		{
			$id_specific_price = $flash_sale[0]['id_specific_price'];
			$this->_deleteSpecificPrice($id_specific_price);
		}
		else
			$id_specific_price = $this->_addSpecificPrice($flash_sale[0]['id_product'], $flash_sale[0]['id_product_attribute'], $flash_sale[0]['date_from'], $flash_sale[0]['date_to'], $flash_sale[0]['discount'], $flash_sale[0]['unit_discount']);

		FsModel::updateStatus($id_flash_sale, $status, $id_specific_price);
	}

	/**
	 * Méthode _deleteTmpCover()
	 * 
	 * Supprime l'ancienne miniature du répertoire /img/tmp
	 * @param type $id_flash_sale
	 * 
	 * @return void
	 */
	private function _deleteTmpCover($id_flash_sale)
	{
		if (file_exists(_PS_TMP_IMG_DIR_.$this->name.'_mini_'.$id_flash_sale.'_1.jpg'))
			unlink(_PS_TMP_IMG_DIR_.$this->name.'_mini_'.$id_flash_sale.'_1.jpg');
	}

	private function _addSpecificPrice($id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $id_currency, $id_country, $id_group)
	{
		$specific_price = new SpecificPrice();
		$specific_price->id_product = (int)$id_product;
		$specific_price->id_product_attribute = (int)$id_product_attribute;
		$specific_price->id_shop = $this->context->shop->id;
		$specific_price->id_currency = (int)$id_currency;
		$specific_price->id_country = (int)$id_country;
		$specific_price->id_group = (int)$id_group;
		$specific_price->id_customer = 0;
		$specific_price->price = -1;
		$specific_price->from_quantity = 1;
		$specific_price->reduction = (float)($unit_discount == 'percentage' ? $discount / 100 : $discount);
		$specific_price->reduction_type = $unit_discount;
		$specific_price->from = $date_from;
		$specific_price->to = $date_to;
		if (!$specific_price->add())
			$this->errors[] = Tools::displayError('An error occurred while adding the specific price.');

		$specific_prices = FsModel::getByProductId((int)$id_product, (int)$id_product_attribute);

		return $specific_prices[0]['id_specific_price'];
	}

	private function _updateSpecificPrice($id_specific_price, $id_product, $id_product_attribute, $date_from, $date_to, $discount, $unit_discount, $id_currency, $id_country, $id_group)
	{
		$specific_price = new SpecificPrice($id_specific_price);
		$specific_price->id_product = (int)$id_product;
		$specific_price->id_product_attribute = (int)$id_product_attribute;
		$specific_price->id_shop = $this->context->shop->id;
		$specific_price->id_currency = (int)$id_currency;
		$specific_price->id_country = (int)$id_country;
		$specific_price->id_group = (int)$id_group;
		$specific_price->id_customer = 0;
		$specific_price->price = -1;
		$specific_price->from_quantity = 1;
		$specific_price->reduction = (float)($unit_discount == 'percentage' ? $discount / 100 : $discount);
		$specific_price->reduction_type = $unit_discount;
		$specific_price->from = $date_from;
		$specific_price->to = $date_to;
		if (!$specific_price->update())
			$this->errors[] = Tools::displayError('An error occurred while updating the specific price.');
	}

	private function _deleteSpecificPrice($id_specific_price)
	{
		$specific_price = new SpecificPrice((int)$id_specific_price);
		if (!$specific_price->delete())
			$this->errors[] = Tools::displayError('An error occurred while attempting to delete the specific price.');
	}

	private function _deleteAllSpecificPrices()
	{
		$flash_sales = FsModel::getFlashSales();
		foreach ($flash_sales as $flash_sale)
			$this->_deleteSpecificPrice($flash_sale['id_specific_price']);

		return true;
	}

	/**
	 * Méthode renderForm()
	 * 
	 * Affiche le formulaire d'ajout/modification des ventes flashs dans le backoffice
	 * 
	 * @return string HTML du formulaire
	 */
	public function renderForm()
	{
		$id_lang = $this->context->cookie->id_lang;
		$version16 = $this->checkPSVersion('1.6.0.0');
		if (!$version16)
		{
			$this->context->controller->addJqueryPlugin('autocomplete');
			$this->context->controller->addJqueryUI('ui.datepicker');
		}

		$this->context->controller->addJS($this->_path.'js/fs-admin.js');

		if (Tools::isSubmit('addflashsales'))
			$action_form = 'add';
		else if (Tools::isSubmit('updateflashsales'))
			$action_form = 'update';

		$id_flash_sale = (Tools::getValue('id_flash_sale') != 0)?Tools::getValue('id_flash_sale'):0;
		$id_specific_price = 0;
		$flash_sale = array();
		$combinations = array();
		if ($id_flash_sale != 0)
		{
			$flash_sale = FsModel::getFlashSale($id_flash_sale);
			$flash_sale = $flash_sale[0];
			$product = new Product($flash_sale['id_product'], true, (int)$id_lang);
			$flash_sale['name'] = $product->name;
			$id_specific_price = ($flash_sale['status'])?$flash_sale['id_specific_price']:0;

			$attributes = $product->getAttributesGroups((int)$id_lang);
			foreach ($attributes as $attribute)
			{
				$combinations[$attribute['id_product_attribute']]['id_product_attribute'] = $attribute['id_product_attribute'];
				if (!isset($combinations[$attribute['id_product_attribute']]['attributes']))
					$combinations[$attribute['id_product_attribute']]['attributes'] = '';
				$combinations[$attribute['id_product_attribute']]['attributes'] .= $attribute['attribute_name'].' - ';

				$combinations[$attribute['id_product_attribute']]['price'] = Tools::displayPrice(
					Tools::convertPrice(
						Product::getPriceStatic((int)$product->id, false, $attribute['id_product_attribute']),
						$this->context->currency
					), $this->context->currency
				);
			}
			foreach ($combinations as &$combination)
				$combination['attributes'] = rtrim($combination['attributes'], ' - ');
		}
		else if (Tools::isSubmit('submit'.$this->name))
		{
			$flash_sale = array(
				'id_currency' => Tools::getValue('id_currency'),
				'id_country' => Tools::getValue('id_country'),
				'id_group' => Tools::getValue('id_group'),
				'date_from' => Tools::getValue('date_from'),
				'date_to' => Tools::getValue('date_to'),
				'discount' => Tools::getValue('discount'),
				'unit_discount' => Tools::getValue('unit_discount')
			);
		}

		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'action_form' => $action_form,
					'id_flash_sale' => $id_flash_sale,
					'id_specific_price' => $id_specific_price,
					'flash_sale' => $flash_sale,
					'combinations' => $combinations,
					'currencies' => Currency::getCurrenciesByIdShop((int)$this->context->shop->id),
					'countries' => Country::getCountriesByIdShop((int)$this->context->shop->id, (int)$this->context->cookie->id_lang),
					'groups' => Group::getGroups((int)$this->context->cookie->id_lang, (int)$this->context->shop->id),
					'support_link' => AdminController::$currentIndex.'&configure='.$this->name.'&support&token='.Tools::getAdminTokenLite('AdminModules'),
					'back_link' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
					'path' => _MODULE_DIR_.$this->name.'/',
					'iso' => Language::getIsoById($this->context->cookie->id_lang),
					'display_name' => $this->displayName,
					'version' => $this->version,
					'author' => $this->author,
					'contact' => $this->contact,
					'psversion16' => $version16
				)
		);

		if ($this->checkPSVersion('1.6.0.0'))
			$html = $this->display(__FILE__, 'views/templates/admin/edit_form.tpl');
		else
			$html = $this->display(__FILE__, 'views/templates/admin/edit_form15.tpl');

		return $html;
	}

	/**
	 * Méthode renderSettingsForm()
	 * 
	 * Affiche le formulaire de paramètres du module dans le backoffice
	 * 
	 * @return string HTML du formulaire
	 */
	public function renderSettingsForm()
	{
		$version16 = $this->checkPSVersion('1.6.0.0');
		$this->context->controller->addJS($this->_path.'js/fs-admin.js');
		$this->context->controller->addJS(_PS_JS_DIR_.'tiny_mce/tiny_mce.js');
		$this->context->controller->addJS(_PS_JS_DIR_.'tinymce.inc.js');
		$this->context->controller->addJqueryPlugin('colorpicker');
		if (!$version16)
		{
			$this->context->controller->addJqueryPlugin('autocomplete');
			$this->context->controller->addJqueryUI('ui.datepicker');
		}

		$product_text = unserialize(rawurldecode(Configuration::get($this->short_name.'_product_text')));
		$meta_title = unserialize(rawurldecode(Configuration::get($this->short_name.'_meta_title')));
		$meta_description = unserialize(rawurldecode(Configuration::get($this->short_name.'_meta_description')));
		$banner_text = unserialize(rawurldecode(Configuration::get($this->short_name.'_banner_text')));
		$banner_title = unserialize(rawurldecode(Configuration::get($this->short_name.'_banner_title')));

		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'support_link' => AdminController::$currentIndex.'&configure='.$this->name.'&support&token='.Tools::getAdminTokenLite('AdminModules'),
					'back_link' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
					'path' => _MODULE_DIR_.$this->name.'/',
					'iso' => Language::getIsoById($this->context->cookie->id_lang),
					'ad' => __PS_BASE_URI__.basename(_PS_ADMIN_DIR_),
					'display_name' => $this->displayName,
					'version' => $this->version,
					'author' => $this->author,
					'contact' => $this->contact,
					'psversion16' => $version16,
					'languages' => Language::getLanguages(false),
					'defaultFormLanguage' => (int)Configuration::get('PS_LANG_DEFAULT'),
					'font_color' => Configuration::get($this->short_name.'_font_color'),
					'bg_color_start' => Configuration::get($this->short_name.'_bg_color_start'),
					'bg_color_end' => Configuration::get($this->short_name.'_bg_color_end'),
					'product_text' => $product_text,
					'product_font_color' => Configuration::get($this->short_name.'_product_font_color'),
					'disp_border' => Configuration::get($this->short_name.'_disp_border'),
					'border_color' => Configuration::get($this->short_name.'_border_color'),
					'meta_title' => $meta_title,
					'meta_description' => $meta_description,
					'banner_title' => $banner_title,
					'banner_text' => $banner_text,
					'banner_title_color' => Configuration::get($this->short_name.'_banner_title_color'),
					'banner_text_color' => Configuration::get($this->short_name.'_banner_text_color'),
					'block_product_number' => Configuration::get($this->short_name.'_block_product_number')
				)
		);

		if ($this->checkPSVersion('1.6.0.0'))
			$html = $this->display(__FILE__, 'views/templates/admin/settings.tpl');
		else
			$html = $this->display(__FILE__, 'views/templates/admin/settings15.tpl');

		return $html;
	}

	public function renderSupportForm()
	{
		// Envoi des paramètres au template
		$this->context->smarty->assign(
				array(
					'path' => _MODULE_DIR_.$this->name.'/',
					'iso' => Language::getIsoById($this->context->cookie->id_lang),
					'display_name' => $this->displayName,
					'version' => $this->version,
					'author' => $this->author,
					'contact' => $this->contact,
					'back_link' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
					'psversion16' => $this->checkPSVersion('1.6.0.0')
				)
		);

		return $this->display(__FILE__, 'views/templates/admin/support.tpl');
	}

	public function getFlashSalesProducts($order_by = '', $order_way = '')
	{
		$products = array();
		$flash_sales = FsModel::getFlashSales($order_by, $order_way);
		$id_lang = Context::getContext()->language->id;

		foreach ($flash_sales as $flash_sale)
		{
			if (time() > strtotime($flash_sale['date_from']) && time() < strtotime($flash_sale['date_to']) && $flash_sale['status'] && $this->_checkUserAccess($flash_sale))
			{
				$product = New Product($flash_sale['id_product'], true, (int)$id_lang);
				$product->id_product = $flash_sale['id_product'];
				$cover = Product::getCover($flash_sale['id_product']);
				$product->id_image = $cover['id_image'];
				$product->date_to = $flash_sale['date_to'];
				if ($this->checkPSVersion('1.6.0.0'))
				{
					if (($product->cache_default_attribute == $flash_sale['id_product_attribute']) || ($flash_sale['id_product_attribute'] == 0))
						$products[] = Product::getProductProperties((int)$id_lang, get_object_vars($product));
				}
				else
				{
					$product = Product::getProductProperties((int)$id_lang, get_object_vars($product));
					if (!in_array($product, $products))
						$products[] = $product;
				}
			}
		}
		return $products;
	}

	/**
	 * Méthode _checkUserAccess()
	 * 
	 * Contrôle si un utilisateur a accès à une vente flash
	 */
	private function _checkUserAccess($flash_sale)
	{
		// Get current user infos
		$id_currency = (int)$this->context->cookie->id_currency;
		$id_customer = (isset($this->context->customer) ? (int)$this->context->customer->id : 0);
		$id_country = (int)$id_customer ? Customer::getCurrentCountry($id_customer) : Configuration::get('PS_COUNTRY_DEFAULT');
		$customer_groups = Customer::getGroupsStatic($id_customer);

		// Check if user has good rights
		if ($flash_sale['id_currency'] != 0 && $flash_sale['id_currency'] != $id_currency)
			return false;

		if ($flash_sale['id_country'] != 0 && $flash_sale['id_country'] != $id_country)
			return false;

		if ($flash_sale['id_group'] != 0 && !in_array($flash_sale['id_group'], $customer_groups))
			return false;

		return true;
	}

	/**
	 * Méthode _generateCSS()
	 * 
	 * Génère un fichier CSS en fonction des paramètres définits dans le backoffice du module
	 */
	private function _generateCSS()
	{
		foreach (Shop::getShops() as $shop)
		{
			// Récupération des paramètres
			$font_color = Configuration::get($this->short_name.'_font_color', null, null, $shop['id_shop']);
			$disp_border = Configuration::get($this->short_name.'_disp_border', null, null, $shop['id_shop']);
			$border_color = Configuration::get($this->short_name.'_border_color', null, null, $shop['id_shop']);
			$bg_color_start = Configuration::get($this->short_name.'_bg_color_start', null, null, $shop['id_shop']);
			$bg_color_end = Configuration::get($this->short_name.'_bg_color_end', null, null, $shop['id_shop']);
			$product_font_color = Configuration::get($this->short_name.'_product_font_color', null, null, $shop['id_shop']);
			$banner_title_color = Configuration::get($this->short_name.'_banner_title_color', null, null, $shop['id_shop']);
			$banner_text_color = Configuration::get($this->short_name.'_banner_text_color', null, null, $shop['id_shop']);
			$banner_img = Configuration::get($this->short_name.'_banner_img', null, null, $shop['id_shop']);

			if ($this->checkPSVersion('1.6.0.0'))
			{
				$font_size = '24px';
				$font_size_cd = '36px';
			}
			else
			{
				$font_size = '14px';
				$font_size_cd = '22px';
			}

			// Récupération du fichier CSS static
			$css_content = Tools::file_get_contents(dirname(__FILE__).'/css/static.css');

			// Génération du fichier CSS
			$css_content .= '
#flash_sales {';
		if ($disp_border)
			$css_content .= '
	-webkit-box-shadow:inset 1px 1px 10px 1px '.$border_color.';
	box-shadow:inset 1px 1px 10px 1px '.$border_color.';
	border-radius:10px;';

		$css_content .= '
}
#flash_sales p{
	font-size:'.$font_size.';
}
#flash_sales .countdownHolder{
	font-size: '.$font_size_cd.';
}
#flash-sales-cat .content_scene_cat_bg{
	background: url(../img/'.$banner_img.') 0 top no-repeat;
	background-size:contain;
}
#flash-sales-cat .category-name{
	color:'.$banner_title_color.' !important;
}
#flash-sales-cat .rte{
	color:'.$banner_text_color.' !important;
}
#flash_sales p {
	color:'.$product_font_color.';
}
.digit,
#flashsales .digit{
	color:'.$font_color.';
	background-color:'.$bg_color_start.';
}
.digit.static{
	background-image: linear-gradient(bottom, '.$bg_color_end.' 50%, '.$bg_color_start.' 50%);
	background-image: -o-linear-gradient(bottom, '.$bg_color_end.' 50%, '.$bg_color_start.' 50%);
	background-image: -moz-linear-gradient(bottom, '.$bg_color_end.' 50%, '.$bg_color_start.' 50%);
	background-image: -webkit-linear-gradient(bottom, '.$bg_color_end.' 50%, '.$bg_color_start.' 50%);
	background-image: -ms-linear-gradient(bottom, '.$bg_color_end.' 50%, '.$bg_color_start.' 50%);

	background-image: -webkit-gradient(
		linear,
		left bottom,
		left top,
		color-stop(0.5, '.$bg_color_end.'),
		color-stop(0.5, '.$bg_color_start.')
	);
}';

			if ($this->checkPSVersion('1.6.0.0'))
				$filename = 'fs-product-s-'.$shop['id_shop'].'.css';
			else
				$filename = 'fs-product15-s-'.$shop['id_shop'].'.css';

			file_put_contents(dirname(__FILE__).'/css/'.$filename, $css_content, LOCK_EX);
		}
	}

	/**
	 * Méthode checkPSVersion()
	 * 
	 * Compare la version de Prestashop passée en paramètre avec la version courante
	 * 
	 * @param string $version Version à comparer
	 * @param string $compare Sens de la comparaison
	 * 
	 * @return boolean True si la comparaison est vérifiée
	 * 
	 */
	public function checkPSVersion($version = '1.5.0.0', $compare = '>')
	{
		return version_compare(_PS_VERSION_, $version, $compare);
	}

	/**
	 * Méthode hookHeader()
	 * 
	 * Ajoute des fichiers css et/ou js dans la balise <head> de la page Html
	 */
	public function hookHeader()
	{
		$version16 = $this->checkPSVersion('1.6.0.0');

		if ((isset($this->context->controller->php_self) || (Tools::getValue('id_product') != '')))
		{
			if (!$version16)
				$this->context->controller->addCSS($this->_path.'css/fs-block.css');
		}

		if ((isset($this->context->controller->php_self) && $this->context->controller->php_self == 'product') || (Tools::getValue('id_product') != ''))
		{
			if ($version16)
				$this->context->controller->addCSS($this->_path.'css/fs-product-s-'.$this->context->shop->id.'.css');
			else
				$this->context->controller->addCSS($this->_path.'css/fs-product15-s-'.$this->context->shop->id.'.css');

			$this->context->controller->addJS($this->_path.'js/jquery.countdown.js');
			$this->context->controller->addJS($this->_path.'js/fs-product.js');
		}

		if ($version16)
		{
			if (isset($this->context->controller->php_self) && $this->context->controller->php_self == 'index')
			{
				$this->context->controller->addCSS($this->_path.'css/fs-product-s-'.$this->context->shop->id.'.css');
				$this->context->controller->addJS($this->_path.'js/jquery.countdown.js');
				$this->context->controller->addJS($this->_path.'js/flashsales.js');
			}
		}
	}

	public function hookExtraRight()
	{
		$current_product = Tools::getValue('id_product');
		$content_only = Tools::getValue('content_only');
		$flash_sales = FsModel::getFlashSaleByIdProduct($current_product);

		if (!count($flash_sales))
			return;

		$attribute = array();
		foreach ($flash_sales as $flash_sale)
		{
			$attribute[] = array(
				'id_product_attribute' => $flash_sale['id_product_attribute'],
				'date_from' => $flash_sale['date_from'],
				'date_to' => $flash_sale['date_to']
			);
		}

		if (strtotime($flash_sale['date_from']) > time() || !$flash_sale['status'] || !$this->_checkUserAccess($flash_sale))
			return;

		$product_text = unserialize(rawurldecode(Configuration::get($this->short_name.'_product_text')));

		$this->smarty->assign(
			array(
				$this->short_name.'_server_date' => date('D M d Y H:i:s O'),
				$this->short_name.'_attribute' => $attribute,
				$this->short_name.'_content_only' => $content_only,
				$this->short_name.'_product_text' => $product_text[$this->context->cookie->id_lang],
				$this->short_name.'_psversion16' => $this->checkPSVersion('1.6.0.0')
			)
		);

		return $this->display(__FILE__, 'product.tpl');
	}

	public function hookLeftColumn()
	{
		$this->context->smarty->assign(
			array(
				$this->short_name.'_path' => $this->_path,
				$this->short_name.'_psversion16' => $this->checkPSVersion('1.6.0.0'),
				$this->short_name.'_display_link' => $this->context->link->getModuleLink($this->name, 'display'),
				$this->short_name.'_products' => $this->getFlashSalesProducts('date_to', 'ASC'),
				$this->short_name.'_block_product_number' => Configuration::get($this->short_name.'_block_product_number')
			)
		);

		if ($this->checkPSVersion('1.6.0.0'))
			return $this->display(__FILE__, 'block.tpl');
		else
			return $this->display(__FILE__, 'block15.tpl');
	}

	public function hookRightColumn()
	{
		return $this->hookLeftColumn();
	}

	public function hookDisplayHomeTab()
	{
		$this->smarty->assign(
			array(
				$this->short_name.'_server_date' => date('D M d Y H:i:s O'),
				'products' => $this->getFlashSalesProducts('date_to', 'ASC')
			)
		);

		return $this->display(__FILE__, 'tab.tpl');
	}

	public function hookDisplayHomeTabContent()
	{
		return $this->display(__FILE__, 'home.tpl');
	}
}